package 你的包名;

import android.util.Log;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import 你的包名.ApiResponse;
import 你的包名.TokenData;
import 你的包名.UserData;
import 你的包名.utils.EncryptUtils;
import org.json.JSONObject;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;
import java.io.IOException;

public class Api {
    private static final String TAG = "API";
    private static volatile Api instance;
    private final HttpClient httpClient;
    private final int timeout = 60;//请求超时时间(秒)，超过这个时间就算超时
    private String publicKey; // RSA公钥
    private String aesKey;    // AES密钥
    private String token;     // 添加Token

    // 需要使用RSA加密的接口列表
    private final Set<String> RSA_APIS = new HashSet<String>() {{
        add(Constants.ApiName.GET_TOKEN.toString());
        add(Constants.ApiName.USER_LOGIN.toString());
        add(Constants.ApiName.USER_REDUCE_MONEY.toString());
        add(Constants.ApiName.USER_REDUCE_VIP_TIME.toString());
        add(Constants.ApiName.USER_REDUCE_VIP_NUMBER.toString());
        add(Constants.ApiName.GET_VIP_DATA.toString());
        // 添加其他需要RSA加密的接口
    }};
    
    private Api() {
        httpClient = HttpClient.getInstance();
    }
    
    public static Api getInstance() {
        if (instance == null) {
            synchronized (Api.class) {
                if (instance == null) {
                    instance = new Api();
                }
            }
        }
        return instance;
    }

    /**
     * 基础请求方法
     * @param apiName 接口名称
     * @param encryptType 加密类型。1=AES加密，2=RSA混合AES加密
     * @param params 请求参数
     * @param callback 回调接口
     */
    private<T> void request(String apiName ,int encryptType, Map<String, Object> params, ApiCallback<T> callback) {
        try {
            int status = EncryptUtils.genRandomStatus();
            long timestamp = System.currentTimeMillis() / 1000;//10位时间戳
            // 1. 添加公共参数
            params = params != null ? params : new HashMap<>();
            params.put("Time", timestamp);
            params.put("Status", status);
            params.put("Api", apiName);
            
            // 2. 将参数转换为JSON
            String jsonParams = new JSONObject(params).toString();

            Log.i(TAG,"原始请求参数："+jsonParams);

            // 3. 根据加密方式 跟 接口名称选择加密方式并加密数据
            Map<String, String> encryptedData;
            if (encryptType == 1){
                //如果是单纯的AES加密，那要先配置一下
                encryptedData = EncryptUtils.encryptWithAES(jsonParams, aesKey);
                Log.i(TAG,"AES加密参数："+encryptedData);
            }else{
                //如果是RSA混合加密的话，随机生成一个秘钥
                aesKey = EncryptUtils.genAesKey();
                Log.i(TAG,"AES密钥："+aesKey);
                if (RSA_APIS.contains(apiName)) {
                    encryptedData = EncryptUtils.encryptWithRSA(jsonParams, aesKey,publicKey);
                    Log.i(TAG,"RSA加密参数："+encryptedData);
                } else {
                    encryptedData = EncryptUtils.encryptWithAES(jsonParams, aesKey);
                    Log.i(TAG,"AES加密参数："+encryptedData);
                }
            }

            // 4. 构造最终的请求数据
            JSONObject requestBody = new JSONObject();
            requestBody.put("a", encryptedData.get("a"));
            requestBody.put("b", encryptedData.get("b"));

            // 5. 构造请求Headers
            Map<String, String> headers = new HashMap<>();
            headers.put("Accept", "application/json, text/plain, */*");
            headers.put("Accept-Language", "zh-CN,zh;q=0.9");
            headers.put("Cache-Control", "no-cache");
            headers.put("Content-Type", "application/json");
            headers.put("Pragma", "no-cache");
            headers.put("Token", token != null ? token : "");

            Log.i(TAG,"请求头："+headers);

            // 6. 发送异步请求
            httpClient.postAsync(Constants.BASE_URL, requestBody.toString(),headers, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    callback.onFailure(e);
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    try {
                        String result = response.body().string();
                        Log.i(TAG,"原始请求结果:"+result);
                        JSONObject responseJson = new JSONObject(result);
                        String a = responseJson.getString("a");
                        String b = responseJson.getString("b");
                        String decryptedData;
                        if (encryptType == 1){
                            decryptedData = EncryptUtils.decryptWithAES(a,b,aesKey);
                        }else{
                            if (RSA_APIS.contains(apiName)){
                                decryptedData = EncryptUtils.decryptWithRSA(a,b,publicKey);
                                Log.i(TAG,"RSA解密结果:"+decryptedData);
                            }else{
                                decryptedData = EncryptUtils.decryptWithAES(a,b,aesKey);
                                Log.i(TAG,"AES结果结果:"+decryptedData);
                            }
                        }

                        JSONObject jsonResponse = new JSONObject(decryptedData);
                        if (jsonResponse.getInt("Status") != status) {
                            throw new Exception("Status错误");
                        }
                        long serverTime = jsonResponse.getLong("Time");
                        if (Math.abs(serverTime - timestamp) > timeout){
                            throw new Exception("请求时间超时，请检查网络");
                        }
                        // 使用Gson解析响应数据
                        Gson gson = new Gson();
                        Type type = ((ParameterizedType) callback.getClass().getGenericInterfaces()[0]).getActualTypeArguments()[0];
                        ApiResponse<T> apiResponse = new ApiResponse<>();
                        apiResponse.setStatus(jsonResponse.getInt("Status"));
                        apiResponse.setTime(jsonResponse.getLong("Time"));
                        apiResponse.setMsg(jsonResponse.optString("Msg"));
                        if (jsonResponse.has("Data")) {
                            T data = gson.fromJson(jsonResponse.getJSONObject("Data").toString(), type);
                            apiResponse.setData(data);
                        }
                        
                        callback.onSuccess(apiResponse);
                    } catch (Exception e) {
                        callback.onFailure(e);
                    }
                }
            });
        } catch (Exception e) {
            callback.onFailure(e);
        }
    }

    /**
     * 设置RSA公钥
     */
    public void setPublicKey(String publicKey) {
        this.publicKey = publicKey;
    }

    /**
     * 设置AES密钥
     */
    public void setAesKey(String aesKey) {
        this.aesKey = aesKey;
    }

    /**
     * 设置Token
     */
    public void setToken(String token) {
        this.token = token;
    }

    /**
     * 获取Token
     */
    public void getToken(Map<String, Object> params, ApiCallback<TokenData> callback) {
        request(Constants.ApiName.GET_TOKEN.toString(),Constants.ENCRYPT_TYPE, params, callback);
    }

    /**
     * 用户登录
     */
    public void login(Map<String, Object> params, ApiCallback<UserData> callback) {
        request(Constants.ApiName.USER_LOGIN.toString(),Constants.ENCRYPT_TYPE, params, callback);
    }

}

// 添加回调接口
// 修改回调接口为泛型接口
interface ApiCallback<T> {
    void onSuccess(ApiResponse<T> response);
    void onFailure(Throwable throwable);
}


