/*
 * @Author: Hann1balLecter hann1ballecter@foxmail.com
 * @Date: 2025-03-15 00:22:48
 * @LastEditors: Hann1balLecter hann1ballecter@foxmail.com
 * @LastEditTime: 2025-03-15 01:49:06
 * @FilePath: \demo\sdk.dart
 * @Description:飞鸟快验SDK的flutter实现
 *
 * Copyright (c) 2025 by Hann1balLecter, All Rights Reserved.
 */

// ignore_for_file: avoid_print

import 'dart:async' show Timer;
import 'dart:convert';
import 'dart:math' as math;

import 'package:encrypt/encrypt.dart';
import 'package:pointycastle/api.dart';
import 'package:pointycastle/asymmetric/api.dart';
import 'package:pointycastle/asymmetric/pkcs1.dart';
import 'package:pointycastle/asymmetric/rsa.dart';
import 'package:pointycastle/block/aes.dart';
import 'package:pointycastle/block/modes/cbc.dart';
import 'package:dio/dio.dart';
import 'package:flutter/foundation.dart';
import 'package:get/get.dart' hide Response;
import 'package:retry/retry.dart';

class Crypto {
  static final Crypto _instance = Crypto._();
  static Crypto get to => _instance;
  Crypto._();
  static String bytesToHex(Uint8List bytes) {
    final buffer = StringBuffer();
    for (var byte in bytes) {
      buffer.write(byte.toRadixString(16).padLeft(2, '0'));
    }
    return buffer.toString();
  }

  static String bytesToBase64(Uint8List data) => base64Encode(data);

  static Uint8List base64ToBytes(String base64String) =>
      base64Decode(base64String);

  static String base64ToUtf8(String base64String) =>
      utf8.decode(base64ToBytes(base64String));

  static Uint8List utf8ToBytes(String utf8String) =>
      Uint8List.fromList(utf8String.codeUnits);

  static String aes({
    required bool encrypt,
    required Uint8List key,
    required Uint8List iv,
    required Uint8List data,
    Uint8List? aad,
    int? macSize,
  }) {
    if (key.length != 16 && key.length != 24 && key.length != 32) {
      throw ArgumentError(
          'AES key length must be 16 bytes (AES-128), 24 bytes (AES-192) or 32 bytes (AES-256)');
    }

    final AESEngine engine = AESEngine();
    final KeyParameter keyParam = KeyParameter(key);
    final params = ParametersWithIV(keyParam, iv);

    final result = _processBlockMode(
        data: data, params: params, engine: engine, encrypt: encrypt);
    return bytesToBase64(result);
  }

  static Uint8List _processBlockMode({
    required Uint8List data,
    required CipherParameters params,
    required BlockCipher engine,
    required bool encrypt,
  }) {
    final BlockCipher cipher = CBCBlockCipher(engine);
    cipher.init(encrypt, params);

    return encrypt
        ? _encryptBlock(data, cipher, engine.blockSize)
        : _decryptBlock(data, cipher, engine.blockSize);
  }

  static Uint8List _encryptBlock(
      Uint8List data, BlockCipher cipher, int blockSize) {
    Uint8List processData;

    processData = _addPkcs7Padding(data, blockSize);

    final output = Uint8List(processData.length);
    for (var offset = 0; offset < processData.length; offset += blockSize) {
      cipher.processBlock(processData, offset, output, offset);
    }

    return output;
  }

  static Uint8List _decryptBlock(
      Uint8List data, BlockCipher cipher, int blockSize) {
    final output = Uint8List(data.length);
    for (var offset = 0; offset < data.length; offset += blockSize) {
      cipher.processBlock(data, offset, output, offset);
    }

    return _removePadding(output, blockSize);
  }

  static Uint8List _addPkcs7Padding(Uint8List data, int blockSize) {
    final padLength = blockSize - (data.length % blockSize);
    final padded = Uint8List(data.length + padLength);
    padded.setRange(0, data.length, data);
    padded.fillRange(data.length, padded.length, padLength);
    return padded;
  }

  static Uint8List _removePadding(Uint8List paddedData, int blockSize) {
    if (paddedData.isEmpty) return paddedData;
    return _removePkcs7Padding(paddedData, blockSize);
  }

  static Uint8List _removePkcs7Padding(Uint8List paddedData, int blockSize) {
    final padLength = paddedData[paddedData.length - 1];
    if (padLength > blockSize || padLength <= 0) {
      throw StateError('Invalid PKCS7 padding');
    }

    for (int i = paddedData.length - padLength; i < paddedData.length; i++) {
      if (paddedData[i] != padLength) {
        throw StateError('Invalid PKCS7 padding');
      }
    }

    return paddedData.sublist(0, paddedData.length - padLength);
  }

  static String md5(Uint8List data) {
    final digest = Digest('MD5');
    final bytes = digest.process(data);
    return bytesToHex(bytes);
  }

  static String rsa(bool encrypt, Uint8List data, String publicKey) {
    final parser = RSAKeyParser();
    final parsedKey = parser.parse(publicKey);

    final engine = RSAEngine();
    final cipher = PKCS1Encoding(engine);

    cipher.init(
        encrypt, PublicKeyParameter<RSAPublicKey>(parsedKey as RSAPublicKey));

    // 处理数据
    final result = cipher.process(data);

    return bytesToBase64(result);
  }
}

class HFlyBird {
  static final HFlyBird _instance = HFlyBird._();
  static HFlyBird get to => _instance;
  HFlyBird._();

  static const Duration _timeLimit = Duration(seconds: 15);
  static const Duration _heartBeatInterval = Duration(seconds: 45);

  Future<HFlyBird> init({
    required String host,
    required String appId,
    String? port,
    required int status,
    required String rsaPublicKey,
  }) async {
    _host = host;
    _appId = appId;
    _port = port ?? '';
    _status = status;
    _rsaPublicKey = rsaPublicKey;
    // 自己去实现设备哈希
    _deviceHash = Crypto.md5(Crypto.utf8ToBytes('xxxxxxx'));
    await refreshToken();

    // 心跳在实际使用中, 应该直接定义到类成员中.如果token失效应先cancel上一个timer.

    Timer.periodic(_heartBeatInterval, (timer) async {
      await retry(
        maxAttempts: 5,
        () async => await heartBeat(),
        retryIf: (e) => e is HeartBeatException,
      );
    });
    return this;
  }

  final Dio _dio = Dio();
  late final String _deviceHash;

  final String _appVersion = '0.0.1';

  late final String _host;
  late final String _appId;
  late final String? _port;
  late final int _status;
  int get status => _status;
  late final String _rsaPublicKey;
  final Uint8List _iv = Uint8List(16);

  String get _api => "http://$_host:$_port/Api?AppId=$_appId";

  String? _token;
  String? _currentAesKey;

  Map<String, dynamic> _generateData(Map<String, dynamic> data) => {
        "Time": DateTime.now().millisecondsSinceEpoch ~/ 1000,
        "Status": _status,
        ...data
      };

  /// 发送API请求并处理加解密逻辑
  Future<VerifyData> _post(Map<String, dynamic> data) async {
    try {
      final Uint8List aesKey;
      if (_currentAesKey == null) {
        final random = math.Random.secure();
        aesKey = Uint8List(24);
        for (var i = 0; i < 24; i++) {
          aesKey[i] = random.nextInt(256);
        }
      } else {
        aesKey = Crypto.utf8ToBytes(_currentAesKey!);
      }

      final Map<String, dynamic> formattedData = _generateData(data);

      print('请求明文 $formattedData');

      // 使用AES加密数据
      final String encryptedData = Crypto.aes(
        encrypt: true,
        key: aesKey,
        iv: _iv,
        data: Crypto.utf8ToBytes(json.encode(formattedData)),
      );

      // 使用RSA公钥加密AES密钥
      final String encryptedAesKey = Crypto.rsa(
        true,
        _currentAesKey == null
            ? aesKey
            : Crypto.base64ToBytes(Crypto.bytesToBase64(aesKey)),
        _rsaPublicKey,
      );

      final Map<String, dynamic> encryptedDataMap = {
        "a": encryptedData,
        "b": encryptedAesKey,
      };

      print('请求密文 $encryptedDataMap');

      final Response resp =
          await _dio.post(_api, data: encryptedDataMap).timeout(_timeLimit);

      print('响应密文 ${resp.data}');

      try {
        final b = resp.data['b'].toLowerCase();
        final isMd5 = GetUtils.isMD5(b);
        // 如果响应数据是十六进制，则使用当前AesKey解密
        if (isMd5) {
          // 生成本地MD5
          final String localMd5 = Crypto.md5(
              Crypto.utf8ToBytes("${resp.data['a']}$_currentAesKey"));

          if (localMd5 != b) {
            print('md5校验失败 $localMd5 != $b');
            throw Exception('Invalid response');
          }

          print('当前aesKey $_currentAesKey');
          // 使用当前AES密钥解密
          final String decryptedData = Crypto.aes(
            encrypt: false,
            key: Crypto.utf8ToBytes(_currentAesKey!),
            iv: _iv,
            data: Crypto.base64ToBytes(resp.data['a']),
          );

          final data = jsonDecode(Crypto.base64ToUtf8(decryptedData));
          print('md5响应: $decryptedData');

          return VerifyData(
            status: data['Status'] == _status,
            message: data['Msg'] ?? "操作成功",
            data: data,
          );
        }
        // 如果响应数据是RSA加密的，则使用RSA公钥解密出AesKey
        final String decryptedAesKey = Crypto.rsa(
          false,
          Crypto.base64ToBytes(resp.data['b']),
          _rsaPublicKey,
        );

        print('解密aesKey $decryptedAesKey');
        print(resp.data['a']);
        final String decryptedData = Crypto.aes(
          encrypt: false,
          key: Crypto.base64ToBytes(decryptedAesKey),
          iv: _iv,
          data: Crypto.base64ToBytes(resp.data['a']),
        );
        print('rsa响应: $decryptedData');

        final data = jsonDecode(Crypto.base64ToUtf8(decryptedData));

        return VerifyData(
          status: data['Status'] == _status,
          message: data['Msg'] ?? "操作成功",
          data: data,
        );
      } catch (e, stackTrace) {
        print('响应解密失败 $e');
        print('响应解密失败 $stackTrace');
        throw Exception(e);
      }
    } catch (e, stackTrace) {
      print('请求失败 $e');
      print('请求失败 $stackTrace');
      return VerifyData(
        status: false,
        message: e.toString(),
        data: null,
      );
    }
  }

  //================================= 基础API接口 =================================

  /// 获取Token
  Future<VerifyData> refreshToken() async {
    try {
      _token = null;
      _currentAesKey = null;
      _dio.options.headers.clear();

      final VerifyData res = await _post({"Api": "GetToken"});

      if (!res.status) throw Exception(res.message);
      _token = res.data['Data']['Token'];
      _currentAesKey = res.data['Data']['CryptoKeyAes'];
      _dio.options.headers['token'] = _token;
      return res;
    } catch (e) {
      return VerifyData(
        status: false,
        message: e.toString(),
        data: null,
      );
    }
  }

  /// 用户登录
  Future<VerifyData> login({
    required String username,
    required String password,
  }) async {
    final res = await _post({
      "Api": "UserLogin",
      "UserOrKa": username,
      "PassWord": password,
      "Key": _deviceHash,
      "AppVer": _appVersion,
    });
    res.expire = res.status ? res.data['Data']['VipTime'] * 1000 : null;
    return res;
  }

  /// 用户注册
  Future<VerifyData> register({
    required String username,
    required String password,
  }) async =>
      _post({
        "Api": "NewUserInfo",
        "User": username,
        "PassWord": password,
        "Key": _deviceHash,
      });

  /// 获取用户IP
  Future<VerifyData> ip() async => _post({
        "Api": "GetUserIP",
      });

  /// 获取应用基础信息
  Future<VerifyData> info() async => _post({
        "Api": "GetAppInfo",
      });

  /// 用户减少余额
  Future<VerifyData> reduceBalance({
    required double money,
    required String log,
    int agentId = 0,
    double agentMoney = 0,
    String agentMoneyLog = "",
  }) async =>
      _post({
        "Api": "UserReduceMoney",
        "Money": money,
        "Log": log,
        "AgentId": agentId,
        "AgentMoney": agentMoney,
        "AgentMoneyLog": agentMoneyLog,
      });

  /// 用户减少积分
  Future<VerifyData> reduceIntegral({
    required double integral,
    required String log,
    int agentId = 0,
    double agentIntegral = 0,
    String agentIntegralLog = "",
  }) async =>
      _post({
        "Api": "UserReduceVipNumber",
        "VipNumber": integral,
        "Log": log,
        "AgentId": agentId,
        "AgentMoney": agentIntegral,
        "AgentMoneyLog": agentIntegralLog,
      });

  /// 用户减少点数
  Future<VerifyData> reducePoint({
    required int point,
    required String log,
  }) async =>
      _post({
        "Api": "UserReduceVipTime",
        "VipTime": point,
        "Log": log,
      });

  /// 检查服务器连接状态
  Future<VerifyData> serverConnect() async => _post({
        "Api": "IsServerLink",
      });

  /// 检查登录状态
  Future<VerifyData> loginStatus() async => _post({
        "Api": "IsLogin",
      });

  /// 获取VIP数据
  Future<VerifyData> vipData() async => _post({
        "Api": "GetVipData",
      });

  /// 获取应用公告
  Future<VerifyData> notice() async => _post({
        "Api": "GetAppGongGao",
      });

  /// 获取专属变量
  Future<VerifyData> appVariable({
    required String name,
  }) async =>
      _post({
        "Api": "GetAppPublicData",
        "Name": name,
      });

  /// 获取公共变量
  Future<VerifyData> publicVariable({
    required String name,
  }) async =>
      _post({
        "Api": "GetPublicData",
        "Name": name,
      });

  /// 设置公共变量
  Future<VerifyData> setPublicVariable({
    required String name,
    required String value,
  }) async =>
      _post({
        "Api": "SetPublicData",
        "Name": name,
        "Value": value,
      });

  /// 检查版本更新
  Future<VerifyData> checkVersion({
    bool isVersionAll = true,
  }) async =>
      _post({
        "Api": "GetAppVersion",
        "Version": _appVersion,
        "IsVersionAll": isVersionAll,
      });

  /// 获取更新下载地址
  Future<VerifyData> updateUrl() async => _post({
        "Api": "GetAppUpDataJson",
      });

  /// 获取应用主页地址
  Future<VerifyData> homeUrl() async => _post({
        "Api": "GetAppHomeUrl",
      });

  /// 设置设备绑定
  Future<VerifyData> bind({
    required String newKey,
    required String username,
    required String password,
  }) async =>
      _post({
        "Api": "SetAppUserKey",
        "NewKey": newKey,
        "User": username,
        "PassWord": password,
      });

  /// 解除设备绑定
  Future<VerifyData> unbind({
    required String username,
    required String password,
  }) async =>
      _post({
        "Api": "DeleteAppUserKey",
        "User": username,
        "PassWord": password,
      });

  /// 发送用户消息
  Future<VerifyData> sendMessage({
    required int type,
    required String content,
    required String password,
  }) async =>
      _post({
        "Api": "SetNewUserMsg",
        "MsgType": type,
        "Msg": content,
        "PassWord": password,
      });

  /// 获取图形验证码
  Future<VerifyData> captcha({
    required int type,
  }) async =>
      _post({
        "Api": "GetCaptcha",
        "CaptchaType": type,
      });

  /// 获取短信验证码
  Future<VerifyData> smsCaptcha({
    required String mobile,
    String username = "",
  }) async =>
      _post({
        "Api": "GetSMSCaptcha",
        "Phone": mobile,
        "User": username,
      });

  /// 提交验证码
  Future<VerifyData> verifyCaptcha({
    required String code,
    required String id,
  }) async =>
      _post({
        "Api": "SubmitCaptcha",
        "Captcha": code,
        "CaptchaId": id,
      });

  /// 获取绑定信息
  Future<VerifyData> bindInfo() async => _post({
        "Api": "GetAppUserKey",
      });

  /// 检查用户是否存在
  Future<VerifyData> checkUser({
    required String username,
  }) async =>
      _post({
        "Api": "IsUserExist",
        "User": username,
      });

  /// 获取用户软件信息
  Future<VerifyData> userAppInfo() async => _post({
        "Api": "GetAppUserInfo",
      });

  /// 获取用户基础信息
  Future<VerifyData> userInfo() async => _post({
        "Api": "GetUserBaseInfo",
      });

  /// 设置用户基础信息
  Future<VerifyData> updateUserInfo({
    required Map<String, dynamic> info,
  }) async =>
      _post({
        "Api": "SetUserBaseInfo",
        ...info,
      });

  /// 获取系统时间戳
  Future<VerifyData> systemTime() async => _post({
        "Api": "GetServerTime",
      });

  /// 获取用户备注
  Future<VerifyData> userRemark() async => _post({
        "Api": "GetAppUserRemark",
      });

  /// 获取会员到期时间
  Future<VerifyData> expireTime() async => _post({
        "Api": "GetAppUserVipTime",
      });

  /// 退出登录
  Future<VerifyData> logout() async => _post({
        "Api": "UserLoginOut",
      });

  /// 远程退出登录
  Future<VerifyData> remoteLogout({
    required String username,
    required String password,
  }) async =>
      _post({
        "Api": "UserLoginOutEx",
        "User": username,
        "PassWord": password,
      });

  /// 心跳保活
  Future<VerifyData> heartBeat() async {
    final res = await _post({"Api": "HeartBeat"});
    if (!res.status) throw HeartBeatException(res.message, res.data);
    return res;
  }

  /// 通过旧密码修改密码
  Future<VerifyData> changePasswordByOld({
    required String oldPassword,
    required String newPassword,
  }) async =>
      _post({
        "Api": "UserPassWordLostByOldPassWord",
        "OldPassWord": oldPassword,
        "NewPassWord": newPassword,
      });

  /// 通过超级密码修改密码
  Future<VerifyData> changePasswordBySuper({
    required String superPassword,
    required String newPassword,
  }) async =>
      _post({
        "Api": "UserPassWordLostBySuperPassWord",
        "SuperPassWord": superPassword,
        "NewPassWord": newPassword,
      });

  /// 通过手机号修改密码
  Future<VerifyData> changePasswordByMobile({
    required String mobile,
    required String captcha,
    required String newPassword,
  }) async =>
      _post({
        "Api": "UserPassWordLostByPhone",
        "Phone": mobile,
        "Captcha": captcha,
        "NewPassWord": newPassword,
      });

  /// 获取用户余额
  Future<VerifyData> userBalance() async => _post({
        "Api": "GetUserMoney",
      });

  /// 获取用户积分
  Future<VerifyData> userIntegral() async => _post({
        "Api": "GetUserVipNumber",
      });

  /// 获取验证码接口列表
  Future<VerifyData> captchaApis() async => _post({
        "Api": "GetCaptchaApiList",
      });

  /// 使用卡密充值
  Future<VerifyData> recharge({
    required String username,
    required String cdkey,
  }) async =>
      _post({
        "Api": "UseKa",
        "User": username,
        "Ka": cdkey,
      });

  /// 获取动态标记
  Future<VerifyData> getDynamicTag({
    required String name,
  }) async =>
      _post({
        "Api": "GetDynamicCfgMark",
        "MarkName": name,
      });

  /// 设置动态标记
  Future<VerifyData> setDynamicTag({
    required String name,
    required String value,
  }) async =>
      _post({
        "Api": "SetDynamicCfgMark",
        "MarkName": name,
        "MarkValue": value,
      });

  /// 余额兑换积分
  Future<VerifyData> exchangeIntegral({
    required double money,
  }) async =>
      _post({
        "Api": "BuyVipNumberByMoney",
        "Money": money,
      });

  /// 获取可购买卡类列表
  Future<VerifyData> cardTypes() async => _post({
        "Api": "GetBuyKaClassList",
      });

  /// 余额购买充值卡
  Future<VerifyData> purchaseCard({
    required int cardId,
    required int quantity,
  }) async =>
      _post({
        "Api": "BuyKaByMoney",
        "KaId": cardId,
        "Number": quantity,
      });

  /// 获取已购买卡号列表
  Future<VerifyData> purchasedCards() async => _post({
        "Api": "GetBuyKaList",
      });

  /// 获取用户类型列表
  Future<VerifyData> userTypes() async => _post({
        "Api": "GetUserTypeList",
      });

  /// 设置用户类型
  Future<VerifyData> setUserType({
    required int typeId,
  }) async =>
      _post({
        "Api": "SetUserType",
        "TypeId": typeId,
      });

  /// 创建直充订单
  Future<VerifyData> directRechargeOrder({
    required String payMethod,
    required int cardTypeId,
    required int quantity,
    required String returnUrl,
  }) async =>
      _post({
        "Api": "UserOrderKaDirect",
        "PayPass": payMethod,
        "KaClassId": cardTypeId,
        "Number": quantity,
        "ReturnUrl": returnUrl,
      });

  /// 创建购卡订单
  Future<VerifyData> cardOrder({
    required String payMethod,
    required int cardTypeId,
    required int quantity,
    required String returnUrl,
  }) async =>
      _post({
        "Api": "UserOrderKaBuy",
        "PayPass": payMethod,
        "KaClassId": cardTypeId,
        "Number": quantity,
        "ReturnUrl": returnUrl,
      });

  /// 创建充值余额订单
  Future<VerifyData> balanceOrder({
    required String payMethod,
    required double amount,
    required String returnUrl,
  }) async =>
      _post({
        "Api": "UserOrderBuyMoney",
        "PayPass": payMethod,
        "Money": amount,
        "ReturnUrl": returnUrl,
      });

  /// 获取支付通道状态
  Future<VerifyData> paymentChannels() async => _post({
        "Api": "UserChargMoneyGetPayPassStatus",
      });

  /// 创建购买积分订单
  Future<VerifyData> integralOrder({
    required String payMethod,
    required double amount,
    required String returnUrl,
  }) async =>
      _post({
        "Api": "UserOrderBuyVipNumber",
        "PayPass": payMethod,
        "Money": amount,
        "ReturnUrl": returnUrl,
      });

  /// 查询订单状态
  Future<VerifyData> orderStatus({
    required String orderId,
  }) async =>
      _post({
        "Api": "UserOrderGetStatus",
        "OrderId": orderId,
      });

  /// 保存用户云配置
  Future<VerifyData> saveUserConfig({
    required String key,
    required String value,
  }) async =>
      _post({
        "Api": "UserCloudCfgSetData",
        "CfgName": key,
        "CfgValue": value,
      });

  /// 获取用户云配置
  Future<VerifyData> getUserConfig({
    required String key,
  }) async =>
      _post({
        "Api": "UserCloudCfgGetData",
        "CfgName": key,
      });

  /// 设置代理标记
  Future<VerifyData> setAgentMark({
    required bool isAgent,
  }) async =>
      _post({
        "Api": "SetAgentTag",
        "IsAgent": isAgent,
      });

  /// 获取代理云配置
  Future<VerifyData> agentConfig() async => _post({
        "Api": "GetAgentCloudCfg",
      });

  /// 获取卡号详情
  Future<VerifyData> cardInfo({
    required String cdkey,
  }) async =>
      _post({
        "Api": "GetKaInfo",
        "Ka": cdkey,
      });

  /// 获取注册赠送卡
  Future<VerifyData> registerGift() async => _post({
        "Api": "GetRegGiveKa",
      });
}

class VerifyData {
  final bool status;
  final String message;
  final dynamic data;
  int? expire;

  VerifyData({
    required this.status,
    required this.message,
    required this.data,
    this.expire,
  });
}

/// 心跳异常
class HeartBeatException implements Exception {
  final String message;
  final dynamic data;

  HeartBeatException(this.message, [this.data]);

  @override
  String toString() => 'HeartBeatException: $message';
}

void main() async {
  // 采用的例子是飞鸟官方的演示站
  // 如果使用自己的服务器  有域名则无需传port  没有域名则传port
  final flyBird = await HFlyBird.to.init(
      host: 'demo.fnkuaiyan.cn',
      appId: '10002',
      // port: '18888',
      status: 10000,
      rsaPublicKey: '''-----BEGIN PUBLIC KEY-----
MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDSd0VGS5aAEAyzyYB7admm2mGU
R6U5a5in/KsefJol5TkP+uJhcx7DtQrNgG6pMk/hiaSBFDO9dv+FXPoKjyLU+QBo
GzFwOnSHP6WuFEHvVs9b11kOfh+zE/dA7JohaOObYTxE0icpAmB7LWabWLilyge8
JD0GiSCZEiL05LdwswIDAQAB
-----END PUBLIC KEY-----''');
  final login = await flyBird.login(username: '123', password: '123');
  print('登录结果: ${login.data}');

  final register = await flyBird.register(username: '123', password: '123');
  print('注册结果: ${register.data}');

  final ip = await flyBird.ip();
  print('获取IP结果: ${ip.data}');
}
